#########################################################
#
# Function to compute the Wald test statistic, 
# the threshold and the corresponding pvalue
#
#########################################################
library(mvtnorm)
#library(rrcov)
library(normalp)
library(MNM)


# Files needed to run this script:
# - MCV_consist
# - ASVcvVN
# - consistMCD
# - consistS
# - fastS_normality
# - fastS_consistency

# --------- Main functions

Waldtest<-function(data, estim ="class",bdp=0.25, dist="norm", df=NULL, level=0.05,equiradial=1){  
  #-----------------------------
  # INPUTS:
  # data : list of K data matrices (one for each group)
  # estim : location and scatter estimators 
  #     'class' : sample estimate
  #     'IQRmed' : (p=1) IQR and median
  #     'MADmed' : (p=1) MAD and median
  #     'MCD'   : MCD estimate (raw)
  #     'RMCD'  : one-step reweighted MCD estimate
  #     'S'     : S estimate with Tukey's biweight function
  # bdp : breakdown point for the MCD, RMCD or S estimators
  # dist : vector of K distributions
  #     'norm' : multivariate normal
  #     'stud' : multivariate Student
  #     'powerexp' : multivariate power exponential distribution
  #     /!\ The elliptical distributions are defined such that COV(X)= sigma, 
  # df : vector of K degree of freedom for the student and powerexp distributions
  #       NB: for the power exp, df=1 corresponds to the normal distribution
  # level: nominal level
  # equiradial: indicator for the equi-radial density case
  
  # OUTPUTS:
  #	chi_quantile: chi quantile 1-alpha, df= k-1
  #	4 lists, each giving
  #		- $Wstat: the Wald test statistics
  #		- $res : the result of the tests
  #		- $pvalue: the p-value
  #	for (Wald, modWald, Wald_inv, modWald_inv) respectively
  #-----------------------------
  
  K<- length(data)
  p<- ncol(data[[1]])
  n_k<- unlist(lapply(data, nrow))
  N<-sum(n_k)
  tau_i<- n_k/N
  
  if(K<2) stop("At least two samples are required")
  if(is.null(dist)) stop("You have to specify a vector of distributions")
  if(length(dist)!=K & equiradial !=1)stop("You have to specify a vector of distributions")
  if(length(dist)!=K & equiradial ==1) dist<-rep(dist,K)
  if(length(df)!=K & equiradial ==1) df<-rep(df,K)
  
  # Computation of VN MCV (consistent)
  if(p>1){
    CV<- unlist(lapply(1:K,function(k) MCV(data[[k]],estim=estim,bdp=bdp,dist=dist[k],df=df[k])$VN))
  }else if(p==1){
    CV<- unlist(lapply(1:K,function(k) uniCV(data[[k]],estim=estim,dist=dist[k],df=df[k])))
  }
    CVinv<- 1/CV
    
  # Computation of ASV
    vec_ASV<- unlist(lapply(1:K, function(k) ASVcvVN(CV[k],p=p,dist=dist[k],df=df[k],estim=estim,bdp=bdp)))
    ASV<- diag(vec_ASV/tau_i)
    ASVinv<- diag(1/CV^4) %*% ASV
  
  # COMPUTATION OF THE H-MATRIX
    H<- cbind(rep(1,K-1), diag(-1, K-1))
  
  # STATISTIC - DECISION THRESHOLD - PVALUE
    q<- qchisq(1-level,df=K-1)
    W<- N * t(CV) %*% t(H)  	%*% solve(H %*% ASV %*% t(H))	 %*% H %*% CV
    Winv<- N * t(CVinv) %*% t(H) 	%*% solve(H %*% ASVinv %*% t(H))	 %*% H %*% CVinv
  
    if(q< W){res<-1}else{res<-0}
    pvalue<- 1-pchisq(W, df=K-1)
    if(q< Winv){res_inv<-1}else{res_inv<-0}
    pvalue_inv<-1-pchisq(Winv,df=K-1)
  
  # EQUI RADIAL DENSITY CASE / MODIFIED WALD TEST
  if(equiradial=='1') {
    pooledCV<- tau_i%*%CV
    pooledinv<- tau_i%*%CVinv
    ASVpooled<- ASVcvVN(g=pooledCV,p=p,dist=dist[1],df=df[1],estim=estim,bdp=bdp)
    ASVpooledinv<- (pooledinv^4)*ASVcvVN(g=1/pooledinv,p=p,dist=dist[1],df=df[1],estim=estim,bdp=bdp)
    modW<- N *t(CV) %*% t(H) %*% solve(ASVpooled * H %*% diag(1/tau_i) %*% t(H)) %*% H %*% CV
    modWinv<- N *t(CVinv) %*% t(H) %*% solve(ASVpooledinv *H %*% diag(1/tau_i) %*%t(H)) %*% H %*% CVinv
    if(q< modW){res_mod<-1}else{res_mod<-0}
    pvalue_mod<- 1-pchisq(modW, df=K-1)
    if(q< modWinv){res_modinv<-1}else{res_modinv<-0}
    pvalue_modinv<-1-pchisq(modWinv,df=K-1)
  }else{
    modW<-modWinv<-res_mod<-res_modinv<-pvalue_mod<-pvalue_modinv<-NULL
  }
    
  return(list( chi_quantile= q,
               Wstat=c( W, modW, Winv, modWinv),
               res=c( res, res_mod, res_inv, res_modinv),
               pvalue= c( pvalue, pvalue_mod, pvalue_inv, pvalue_modinv)
  ))
}


# SEMI PARAMETRIC TESTS (only for classical estimator)

WaldtestSP<-function(data, level=0.05){  
  #-----------------------------
  # INPUTS:
  # data : list of K data matrices (one for each group)
  # level: nominal level

  # OUTPUTS:
  #	chi_quantile: chi quantile 1-alpha, df= k-1
  #	4 lists, each giving
  #		- $Wstat: the Wald test statistics
  #		- $res : the result of the tests
  #		- $pvalue: the p-value
  #	for (Wald, modWald, Wald_inv, modWald_inv) respectively
  #-----------------------------
  
  K<- length(data)
  p<- ncol(data[[1]])
  n_k<- unlist(lapply(data, nrow))
  N<-sum(n_k)
  tau_i<- n_k/N
  
  if(K<2) stop("At least two samples are required")

  # Computation of VN MCV (consistent)
    kurtosis=NULL
    moy<-list()
    cov<-list()
    CV<-NULL
    
    if(p>1){
      for(k in 1:K){
        moy[[k]]<- apply(data[[k]],2,mean) 
        cov[[k]]<- (n_k[k]-1)/n_k[k] * cov(data[[k]]) 
        CV[k]<- cvvoinov(moy[[k]],cov[[k]])# biased version
        # Computation of the kurtosis   
        kurtosis[k]<- 1/(p*(p+2))* mean(apply(data[[k]],1,DistMahala,mu=moy[[k]],sigma=cov[[k]])^2)-1
      }
    }else if(p==1){
      moy[[k]]<-mean(data[[k]])
      cov[[k]]<- sqrt((n_k[k]-1)/n_k[k]) * sd(data[[k]])   # biased version
      CV[k]<- cov[[k]]/moy[[k]]
      # kurtosis
      kurtosis[k]<- 1/(p*(p+2))*mean(apply(data[[k]],1,function(x,mu,sigma)(x-mu)/sigma,mu=moy[[k]],sigma=as.numeric(cov[[k]]))^4)-1
    }
    tau<- rep(1,K)
    Cte<-  2+3*kurtosis
  vec_ASV<-tau*CV^4 + CV^2/4*Cte  
  CVinv<- 1/CV
  ASV<- diag(vec_ASV/tau_i)
  ASVinv<- diag(1/CV^4) %*% ASV
 
  # COMPUTATION OF THE H-MATRIX
  H<- cbind(rep(1,K-1), diag(-1, K-1))
  
  # STATISTIC - DECISION THRESHOLD - PVALUE
  q<- qchisq(1-level,df=K-1)
  W<- N * t(CV) %*% t(H)  	%*% solve(H %*% ASV %*% t(H))	 %*% H %*% CV
  Winv<- N * t(CVinv) %*% t(H) 	%*% solve(H %*% ASVinv %*% t(H))	 %*% H %*% CVinv
  
  if(q< W){res<-1}else{res<-0}
  pvalue<- 1-pchisq(W, df=K-1)
  if(q< Winv){res_inv<-1}else{res_inv<-0}
  pvalue_inv<-1-pchisq(Winv,df=K-1)
  
  # EQUI RADIAL DENSITY CASE / MODIFIED WALD TEST

    pooledCV<- tau_i%*%CV
    pooledinv<- tau_i%*%CVinv
    ASVpooled<- as.numeric(tau[1]*pooledCV^4 + Cte[1]*pooledCV^2/4)
    ASVpooledinv<- as.numeric(pooledinv^4*(tau[1]*pooledinv^(-4)+ Cte[1]*pooledinv^(-2)/4))
    modW<- N *t(CV) %*% t(H) %*% solve(ASVpooled * H %*% diag(1/tau_i) %*% t(H)) %*% H %*% CV
    modWinv<- N *t(CVinv) %*% t(H) %*% solve(ASVpooledinv *H %*% diag(1/tau_i) %*%t(H)) %*% H %*% CVinv
    if(q< modW){res_mod<-1}else{res_mod<-0}
    pvalue_mod<- 1-pchisq(modW, df=K-1)
    if(q< modWinv){res_modinv<-1}else{res_modinv<-0}
    pvalue_modinv<-1-pchisq(modWinv,df=K-1)
 
  return(list( chi_quantile= q,
               Wstat=c( W, modW, Winv, modWinv),
               res=c( res, res_mod, res_inv, res_modinv),
               pvalue= c( pvalue, pvalue_mod, pvalue_inv, pvalue_modinv)
  ))
}


